"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getKey = void 0;

var _symmetricKey = _interopRequireDefault(require("./symmetric-key"));

var _lruCache = _interopRequireDefault(require("lru-cache"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

// This code is based on the `mssql-jdbc` library published under the conditions of MIT license.
// Copyright (c) 2019 Microsoft Corporation
const cache = new _lruCache.default(0);

const getKey = async (keyInfo, options) => {
  if (!options.trustedServerNameAE) {
    throw new Error('Server name should not be null in getKey');
  }

  const serverName = options.trustedServerNameAE;
  const keyLookupValue = `${serverName}:${Buffer.from(keyInfo.encryptedKey).toString('base64')}:${keyInfo.keyStoreName}`;

  if (cache.has(keyLookupValue)) {
    return cache.get(keyLookupValue);
  } else {
    const provider = options.encryptionKeyStoreProviders && options.encryptionKeyStoreProviders[keyInfo.keyStoreName];

    if (!provider) {
      throw new Error(`Failed to decrypt a column encryption key. Invalid key store provider name: ${keyInfo.keyStoreName}. A key store provider name must denote either a system key store provider or a registered custom key store provider. Valid (currently registered) custom key store provider names are: ${options.encryptionKeyStoreProviders}. Please verify key store provider information in column master key definitions in the database, and verify all custom key store providers used in your application are registered properly.`);
    }

    const plaintextKey = await provider.decryptColumnEncryptionKey(keyInfo.keyPath, keyInfo.algorithmName, keyInfo.encryptedKey);
    const encryptionKey = new _symmetricKey.default(plaintextKey);

    if (options.columnEncryptionKeyCacheTTL > 0) {
      cache.set(keyLookupValue, encryptionKey, options.columnEncryptionKeyCacheTTL);
    }

    return encryptionKey;
  }
};

exports.getKey = getKey;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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