"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
const MAX = (1 << 16) - 1;
const UNKNOWN_PLP_LEN = Buffer.from([0xfe, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff]);
const PLP_TERMINATOR = Buffer.from([0x00, 0x00, 0x00, 0x00]);
const NULL_LENGTH = Buffer.from([0xFF, 0xFF]);
const MAX_NULL_LENGTH = Buffer.from([0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF]);
const NVarChar = {
  id: 0xE7,
  type: 'NVARCHAR',
  name: 'NVarChar',
  maximumLength: 4000,
  declaration: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    let length;

    if (parameter.length) {
      length = parameter.length;
    } else if (value != null) {
      length = value.toString().length || 1;
    } else if (value === null && !parameter.output) {
      length = 1;
    } else {
      length = this.maximumLength;
    }

    if (length <= this.maximumLength) {
      return 'nvarchar(' + length + ')';
    } else {
      return 'nvarchar(max)';
    }
  },
  resolveLength: function (parameter) {
    const value = parameter.value; // Temporary solution. Remove 'any' later.

    if (parameter.length != null) {
      return parameter.length;
    } else if (value != null) {
      if (Buffer.isBuffer(value)) {
        return value.length / 2 || 1;
      } else {
        return value.toString().length || 1;
      }
    } else {
      return this.maximumLength;
    }
  },

  generateTypeInfo(parameter) {
    const buffer = Buffer.alloc(8);
    buffer.writeUInt8(this.id, 0);

    if (parameter.length <= this.maximumLength) {
      buffer.writeUInt16LE(parameter.length * 2, 1);
    } else {
      buffer.writeUInt16LE(MAX, 1);
    }

    if (parameter.collation) {
      parameter.collation.toBuffer().copy(buffer, 3, 0, 5);
    }

    return buffer;
  },

  generateParameterLength(parameter, options) {
    if (parameter.value == null) {
      if (parameter.length <= this.maximumLength) {
        return NULL_LENGTH;
      } else {
        return MAX_NULL_LENGTH;
      }
    }

    let value = parameter.value;

    if (parameter.length <= this.maximumLength) {
      let length;

      if (value instanceof Buffer) {
        length = value.length;
      } else {
        value = value.toString();
        length = Buffer.byteLength(value, 'ucs2');
      }

      const buffer = Buffer.alloc(2);
      buffer.writeUInt16LE(length, 0);
      return buffer;
    } else {
      return UNKNOWN_PLP_LEN;
    }
  },

  *generateParameterData(parameter, options) {
    if (parameter.value == null) {
      return;
    }

    let value = parameter.value;

    if (parameter.length <= this.maximumLength) {
      if (value instanceof Buffer) {
        yield value;
      } else {
        value = value.toString();
        yield Buffer.from(value, 'ucs2');
      }
    } else {
      if (value instanceof Buffer) {
        const length = value.length;

        if (length > 0) {
          const buffer = Buffer.alloc(4);
          buffer.writeUInt32LE(length, 0);
          yield buffer;
          yield value;
        }
      } else {
        value = value.toString();
        const length = Buffer.byteLength(value, 'ucs2');

        if (length > 0) {
          const buffer = Buffer.alloc(4);
          buffer.writeUInt32LE(length, 0);
          yield buffer;
          yield Buffer.from(value, 'ucs2');
        }
      }

      yield PLP_TERMINATOR;
    }
  },

  validate: function (value) {
    if (value == null) {
      return null;
    }

    if (typeof value !== 'string') {
      throw new TypeError('Invalid string.');
    }

    return value;
  }
};
var _default = NVarChar;
exports.default = _default;
module.exports = NVarChar;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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